package fips.game.jchess;

import java.util.logging.Logger;

/** Simple engine implementation, just tries out all possible
 *  moves and evaluates them.
 *
 *  The calculation is done in a different thread and controlled
 *  with the haltflag.
 *
 */ 
public class ChessEngine implements Runnable, Player {

/** If 0 this is the current calculation leve, if 1
 *  descend to next level.
 */
  final static int	DEPTH_POS	= 64;
  final static int	CURRMOVE_POS	= 65;
  final static int	NEXTMOVE_POS	= 66;
  final static int	BESTMOVE_POS	= 67;
  final static int	BESTVAL_POS	= 68;
  final static int	MOVESUM_POS	= 69;	// number of all evaluated bds


  Board			board;

/** Color of the engine
 */
  int			color;

/** Color of the current mover.
 */
  int			moving;


/** Thread for calculations
 */
  Thread		thread;


/** Boolean haltflag. If true calculation halted
 */
  private boolean	haltflag;





// Limitations
/** If true the chessengine won't calculate moves while the opponent
 *  is thinking.
 */
  boolean		easy;

/** If true the chessengine uses a time limit for calculating
 *  moves.
 */
  boolean		withclock=false;
  int			maxdepth;
  int			maxmoves;
  long			endtime;


/** Best move found so far for curdepth-1 (if running) or curdepth
 *  (if calculation complete).
 */
  int			bestmove;
  int			bestval;



  Logger		logger;



/** Create a new chess engine playing on the given board
 *  using the given color.
 */
  public ChessEngine() {

    logger=fips.util.LogManager.getLogger("jchess.ChessEngine");

    this.color=-1;
    this.board=null;

// basic
    moving=-1;

// limits
    easy=true;
    withclock=false;
    maxdepth=40;
//    maxmoves=-1;
    maxmoves=500000;
    endtime=-1;

    haltflag=true;
    thread=new Thread(this);
    thread.start();
  }


//**************************************************************************//
//
//			Methods to configure the engine			    //
//
//**************************************************************************//










//**************************************************************************//
//
//			Interface implementation
//
//**************************************************************************//




/** BoardListener implementation to get notification when board changed.
 *  This may only happen when opponent made move, so the calculation
 *  may only be running when mover is opponents color and easy
 *  is enabled.
 */
  public void boardChanged(Board board) {
    logger.fine(".boardChanged() entered.");

    this.board=board;
    moving=board.getMover();

    logger.finer("New mover is "+moving);

    startEngine();
  }


  public void sit(Board board, int color) {
    if((this.board!=null)&&(board!=null)&&(this.board!=board)) {
      throw new IllegalStateException();
    }

    this.board=board;
    this.color=color;
    moving=board.getMover();

    if((moving==color)&&(board.gameStarted())) {
      //ERROR ("#: no startup yet");
    }
  }






//**************************************************************************//
//
//			Move calculation
//
//********************************



  public void startEngine() {
    logger.fine(".startEngine()");

    synchronized(thread) {
      haltflag=false;
      thread.notifyAll();
    }
  }



  public void run() {
    int bdlist[][];
    bdlist=new int[maxdepth+1][];
    for(int i=0; i<bdlist.length; i++) {
      bdlist[i]=new int[72];
    }


// the active mover when calculation started
    int calcmover=0;

// the depth, up to that the calculation was completed,
// e.g. 1 means just one move
    int resultdepth=0;

// if true, engine will make its move
    boolean makemove=false;

    //TRACE ("#: starting calculation thread ...");
    while(true) {
      synchronized(thread) {
        while((haltflag)||(easy&&(color!=moving))) {
          logger.finer("Entered halt state ...");
          try {
            thread.wait();
          } catch (InterruptedException ie) {}
        }
      }


      if(calcmover==moving) {
// check if the board is still the same, if so continue
// with already started calculation

        logger.severe("No reuse check for cont");
        int data[]=board.getData();

        System.arraycopy(data, 0, bdlist[0], 0, data.length);
        resultdepth=0;
        bestmove=-1;

        initCalcVars(bdlist, 0);
        bdlist[0][MOVESUM_POS]=0;
      } else {
// try to use calculation data from before

        logger.severe("No reuse check for easy mode");

        int data[]=board.getData();

        System.arraycopy(data, 0, bdlist[0], 0, data.length);
        resultdepth=0;
        bestmove=-1;

        initCalcVars(bdlist, 0);
        bdlist[0][MOVESUM_POS]=0;
        calcmover=moving;
      }


      if(calcmover==color) {

        while((!haltflag)&&(resultdepth<maxdepth)&&(!makemove)) {

          logger.fine("Doing calculations for depth "+resultdepth+
                      ", moves so far "+bdlist[0][MOVESUM_POS]);
          optimizeMove(bdlist, resultdepth+1, calcmover);
          if(bdlist[0][NEXTMOVE_POS]==Integer.MAX_VALUE) {
// calculation finished, increase resultdepth
            bestmove=bdlist[0][BESTMOVE_POS];
            bestval=bdlist[0][BESTVAL_POS];

            logger.fine("Best move after "+bdlist[0][MOVESUM_POS]+
                        " moves for depth "+resultdepth+" is 0x"+
                        Integer.toHexString(bestmove)+", value "+bestval);

            initCalcVars(bdlist, 0);
            resultdepth++;

          } else {
            logger.fine("External stop detected");

// calculation stopped (timeout, moves exceeded??)

// optimize until mover==color

          }
          if((resultdepth==maxdepth)||(bdlist[0][MOVESUM_POS]==maxmoves)) {
            logger.fine("Making move: depth is "+resultdepth+", moves "+
                        bdlist[0][MOVESUM_POS]);

            makemove=true;
          }
        } // end of optimize loop while

      } else {
        throw new InternalError("Not implemented");
      }

      if(makemove) {
        logger.fine("Making move for "+calcmover+", mine "+color);
        if(calcmover==color) {
          if(bestmove>=0) {

// transform bestmove to move object
            board.makeMove(this, createMove(bestmove, bdlist[0][bestmove>>8]));

            if(easy) haltflag=true;
          } else {
            throw new InternalError("Giving up");
          }
        }
        makemove=false;
      }

    }
  }


  private void initCalcVars(int bdlist[][], int depth) {
    bdlist[depth][DEPTH_POS]=0;
    bdlist[depth][CURRMOVE_POS]=-1;
    bdlist[depth][NEXTMOVE_POS]=0;

    bdlist[depth][BESTMOVE_POS]=-1;
    bdlist[depth][BESTVAL_POS]=0;
    if(depth!=0) bdlist[depth][MOVESUM_POS]=bdlist[depth-1][MOVESUM_POS];
  }





/** Optimize moves on a given board. The optimization starts
 *  with nextmove. 
 *
 *
 *  @depth current depth, where to continue eval
 *  @param optcolor color of the mover with the first move
 *  @returns true if optimization terminated normally, false
 *  if timeout or halt occured.
 */
  void optimizeMove(int bdlist[][], int depthlimit, int currmover) {

    int currdepth=0;
    while(bdlist[currdepth][DEPTH_POS]==1) {
      currmover^=ChessDefs.COLOR;
      currdepth++;
    }

    while((!haltflag)&&
          ((maxmoves<=0)||(bdlist[currdepth][MOVESUM_POS]<maxmoves))&&
          ((!withclock)||(System.currentTimeMillis()<endtime))) {
// now at depth where DEPTH_POS=0


      while(bdlist[currdepth][NEXTMOVE_POS]==Integer.MAX_VALUE) {
// decrease depth

//        logger.fine("Decreasing depth at "+currdepth);

        currdepth--;
        currmover^=ChessDefs.COLOR;
        if(currdepth<0) break;	// no decrease possible

        if(currmover==ChessDefs.WHITE) {
// maximize val
          if((bdlist[currdepth][BESTMOVE_POS]==-1)||
             (bdlist[currdepth][BESTVAL_POS]<bdlist[currdepth+1][BESTVAL_POS])) {
            bdlist[currdepth][BESTMOVE_POS]=bdlist[currdepth][CURRMOVE_POS];
            bdlist[currdepth][BESTVAL_POS]=bdlist[currdepth+1][BESTVAL_POS];
          }
        } else {
// minimize val
          if((bdlist[currdepth][BESTMOVE_POS]==-1)||
             (bdlist[currdepth][BESTVAL_POS]>bdlist[currdepth+1][BESTVAL_POS])) {
            bdlist[currdepth][BESTMOVE_POS]=bdlist[currdepth][CURRMOVE_POS];
            bdlist[currdepth][BESTVAL_POS]=bdlist[currdepth+1][BESTVAL_POS];
          }
        }


        bdlist[currdepth][MOVESUM_POS]=bdlist[currdepth+1][MOVESUM_POS];
        bdlist[currdepth][DEPTH_POS]=0;
      }
      if(currdepth<0) break;


// make the move
//      logger.fine("Making move "+bdlist[currdepth][NEXTMOVE_POS]);
      if(!makeMove(bdlist[currdepth][NEXTMOVE_POS], currmover,
                   bdlist[currdepth], bdlist[currdepth+1])) {
//        logger.fine("Move is illegal");
//        bdlist[currdepth][MOVESUM_POS]++;

        continue;
      }

      bdlist[currdepth][MOVESUM_POS]++;


// move made, check it
/*
      for(int i=0; i<64; i++) {
        if(((bdlist[currdepth+1][i]&ChessDefs.PIECE)==ChessDefs.KING)&&
           ((bdlist[currdepth+1][i]&ChessDefs.COLOR)==currmover)&&
           (ChessDefs.isAttacked(i, bdlist[currdepth+1],
                                 currmover^ChessDefs.COLOR))) {
        }
      }
*/

      int score=evaluateBoard(bdlist[currdepth+1]);


      if((currdepth+1<depthlimit)&&(score>-MATE_POINTS)&&
         (score<MATE_POINTS)) {
// increase depth
//        logger.fine("Increasing depth from "+currdepth);

        bdlist[currdepth][DEPTH_POS]=1;
        currdepth++;
        initCalcVars(bdlist, currdepth);
        currmover^=ChessDefs.COLOR;
      } else {
// write this boards score into the the eval list at currdepth

        if(bdlist[currdepth][BESTMOVE_POS]<0) {
          bdlist[currdepth][BESTMOVE_POS]=bdlist[currdepth][CURRMOVE_POS];
          bdlist[currdepth][BESTVAL_POS]=score;
        } else {

// check if move is better or worse

          if(currmover==ChessDefs.WHITE) {
// maximize val
            if(bdlist[currdepth][BESTVAL_POS]<score) {
              bdlist[currdepth][BESTMOVE_POS]=bdlist[currdepth][CURRMOVE_POS];
              bdlist[currdepth][BESTVAL_POS]=score;
            }
          } else {
// minimize val
            if(bdlist[currdepth][BESTVAL_POS]>score) {
              bdlist[currdepth][BESTMOVE_POS]=bdlist[currdepth][CURRMOVE_POS];
              bdlist[currdepth][BESTVAL_POS]=score;
            }
          }
        }
      }
    }

    while(currdepth>0) {
      currdepth--;
      bdlist[currdepth][MOVESUM_POS]=bdlist[currdepth+1][MOVESUM_POS];
    }
  }


// both dir arrays will give all combinations with with x=dir[i], y=dir[i+2] 

  private final static int dirs[] = {0, 1, 1, 1, 0, -1, -1, -1};
  private final static int knightmoves[] = {2, 1, -1, -2, -2, -1, 1, 2};



/** Make a move for one player on a given board and write
 * result to the destination board. The method doesn't check
 * for illegal king moves or draw.
 * @returns true if the given move is possible
 */

  private boolean makeMove(int move, int color, int srcb[], int dstb[]) {
    int i, j, k;
    int x, y, nx, ny;

    srcb[CURRMOVE_POS]=move;

    int pos=move>>8;

    int piece=srcb[pos];
    if(((piece&ChessDefs.PIECE)==0)||((piece&ChessDefs.COLOR)!=color)) {
      srcb[NEXTMOVE_POS]=(pos==63?Integer.MAX_VALUE:(pos+1)<<8);
      return(false);
    }


// move enumeration count, for each piece different
    int moveenum=move&0xFF;

    System.arraycopy(srcb, 0, dstb, 0, 64);
    x=pos&7;
    y=(pos>>3)&7;

// points to the next move for the selected piece or 0x100
// for next piece 
// init to -1 to get ArrayIndexOutOfBounds if uninitialized
    int nextmove=-1;


    int endpos=-1;
    i=-1;

    switch(piece&ChessDefs.PIECE) {
//
// 0..1forward   1..attack right   2..attack left   3..2forward
// 4..forward+K  5..right+K        6..left+K
//
      case ChessDefs.PAWN:
        boolean toknight=false;

        i=((color==ChessDefs.WHITE)?8:-8);
        switch(moveenum) {
          case 0: // 1 forwd
          case 4: // 1 forwd+K
            if((moveenum==4)&&(((i>0)&&(x!=6))||((i<0)&&(x!=1)))) {
              nextmove=0x100;
            } else {
              nextmove=moveenum+1;
              if(srcb[pos+i]==0) endpos=pos+i;
              if(moveenum==4) toknight=true;
            }
            break;
          case 1: // attack right
          case 5:
            nextmove=moveenum+1;
            if(x==7) break;

            if((srcb[pos+i+1]==0)||((srcb[pos+i+1]&ChessDefs.COLOR)==color)) {
              if((srcb[pos+i+1]==0)&&((srcb[pos+1]&ChessDefs.PIECE)==ChessDefs.PAWN)&&
                 ((srcb[pos+1]&ChessDefs.MOVED2)!=0)&&((srcb[pos+1]&ChessDefs.COLOR)!=color)) {
                endpos=pos+i+1;
                dstb[pos+1]=0;		// kill pawn en passent
              }
            } else endpos=pos+i+1;
            if((moveenum==5)&&(((i>0)&&(x==6))||((i<0)&&(x==1)))) toknight=true;
            break;
          case 2: // attack left
          case 6:
            nextmove=((moveenum==6)?0x100:moveenum+1);
            if(x==0) break;

            if((srcb[pos+i-1]==0)||((srcb[pos+i-1]&ChessDefs.COLOR)==color)) {
              if((srcb[pos+i-1]==0)&&((srcb[pos-1]&ChessDefs.PIECE)==ChessDefs.PAWN)&&
                 ((srcb[pos-1]&ChessDefs.MOVED2)!=0)&&((srcb[pos-1]&ChessDefs.COLOR)!=color)) {
                endpos=pos+i-1;
                dstb[pos-1]=0;          // kill pawn en passent
              }
            } else endpos=pos+i-1;
            if((moveenum==6)&&(((i>0)&&(x==6))||((i<0)&&(x==1)))) toknight=true;
            break;
          case 3: // 2 forward
            if((((i>0)&&(y==1))||((i<0)&&(y==6)))&&
               (srcb[pos+i]==0)&&(srcb[pos+i+i]==0)) {
              endpos=pos+i+i;
            }
            nextmove=moveenum+1;
            break;
          default:
            throw new InternalError("Invalid variant "+moveenum);
        }

        if(endpos<0) break;

        if(moveenum==3) dstb[pos]|=ChessDefs.MOVED2; 

        i=endpos>>3;
        if((i==0)||(i==7)) { // promote pawn
          dstb[pos]=(toknight ? ChessDefs.KNIGHT : ChessDefs.QUEEN) | ChessDefs.MOVED | color;
        }
        break;


//
//
//   
      case ChessDefs.BISHOP:
// which diagonal line to use 
        int diagsel=(moveenum>>2)|1;
        int dist=moveenum&0x7;

        nx=x+(1+dist)*dirs[diagsel];
        ny=y+(1+dist)*dirs[(diagsel+2)&0x7];
        endpos=nx+(ny<<3);

        if((nx<0)||(nx>7)||(ny<0)||(ny>7)||
           ((srcb[endpos]!=0)&&((srcb[endpos]&ChessDefs.COLOR)==color))) {
          endpos=-1;
        } else {
// check if way is free
          k=pos;
          for(int dcnt=0; dcnt<dist; dcnt++) {
            k+=dirs[diagsel]+(dirs[(diagsel+2)&0x7]<<3);
            if(srcb[k]!=0) {
              endpos=-1;
              break;
            }
          }
        }
        if(endpos<0) nextmove=((diagsel==7)?0x100:((diagsel+2)<<2));
        else nextmove=moveenum+1;
        break;




//
// 0.. 2up,1right  7..2up,1left
//
      case ChessDefs.KNIGHT:
        if(moveenum>7) nextmove=0x100;
        else {
          nx=x+knightmoves[moveenum];
          ny=y+knightmoves[(moveenum+2)&7];
         
          nextmove=(moveenum==7?0x100:moveenum+1);

          if((nx>=0)&&(nx<8)&&(ny>=0)&&(ny<8)) {
            endpos=nx+(ny<<3);

            if((endpos>=0)&&(srcb[endpos]!=0)&&
               ((srcb[endpos]&ChessDefs.COLOR)==color)) endpos=-1;
          }
        }
        break;

      case ChessDefs.ROOK:
        int direction=(moveenum>>2)&(-2);
        dist=moveenum&0x7;

        if((dist<7)&&(direction<7)) {
          nx=x+(1+dist)*dirs[direction];
          ny=y+(1+dist)*dirs[(direction+2)&0x7];
          endpos=nx+(ny<<3);

          if((nx<0)||(nx>7)||(ny<0)||(ny>7)||
             ((srcb[endpos]!=0)&&((srcb[endpos]&ChessDefs.COLOR)==color))) {
            endpos=-1;
          } else {
// check if way is free
            k=pos;
            for(int dcnt=0; dcnt<dist; dcnt++) {
              k+=dirs[direction]+(dirs[(direction+2)&0x7]<<3);
              if(srcb[k]!=0) {
                endpos=-1;
                break;
              }
            }
          }
        }

        if(endpos<0) nextmove=((direction>=6)?0x100:((direction+2)<<2));
        else nextmove=moveenum+1;
        break;

//
//
//
     case ChessDefs.QUEEN:

       direction=(moveenum>>3);
       dist=moveenum&0x7;

       if((dist<7)&&(direction<8)) {
         nx=x+(1+dist)*dirs[direction];
         ny=y+(1+dist)*dirs[(direction+2)&0x7];
         endpos=nx+(ny<<3);
         if((nx<0)||(nx>7)||(ny<0)||(ny>7)||
            ((srcb[endpos]!=0)&&((srcb[endpos]&ChessDefs.COLOR)==color))) {
           endpos=-1;
         } else {
// check if way is free
           k=pos;
           for(int dcnt=0; dcnt<dist; dcnt++) {
             k+=dirs[direction]+(dirs[(direction+2)&0x7]<<3);
             if(srcb[k]!=0) {
               endpos=-1; break;
             }
           }
         }
       }
       if(endpos<0) nextmove=((direction>=7)?0x100:((direction+1)<<3));
       else nextmove=moveenum+1;
       break;


//
// move the king 0-7 dir, 8 rochade right, 9 rochade left
//
      case ChessDefs.KING:
        if(moveenum<8) {
          nx=x+(dirs[moveenum]);
          ny=y+(dirs[(moveenum+2)&0x7]);
          endpos=nx+(ny<<3);
 
          if((nx<0)||(nx>7)||(ny<0)||(ny>7)||
             (((srcb[endpos]&ChessDefs.PIECE)!=0)&&
              ((srcb[endpos]&ChessDefs.COLOR)==color))) {
            endpos=-1;
          }
          nextmove=moveenum+1;
        } else if(moveenum<10) { // do rochades
          if(((piece&ChessDefs.MOVED)!=0)||((pos!=4)&&(pos!=60))||
             (ChessDefs.isAttacked(pos, srcb, color^ChessDefs.COLOR))) {
            nextmove=0x100;
          } else {
            if(moveenum==8) { // to the right
              if((srcb[pos+1]==0)&&(srcb[pos+2]==0)&&
                 ((srcb[pos+3]&ChessDefs.PIECE)==ChessDefs.ROOK)&&
                 ((srcb[pos+3]&ChessDefs.MOVED)==0)&&
                 ((srcb[pos+3]&ChessDefs.COLOR)==color)&&
                 (!ChessDefs.isAttacked(pos+1, srcb, color^ChessDefs.COLOR))&&
                 (!ChessDefs.isAttacked(pos+2, srcb, color^ChessDefs.COLOR))&&
                 (!ChessDefs.isAttacked(pos+3, srcb, color^ChessDefs.COLOR))) {
                endpos=pos+2;

                dstb[pos+1]=srcb[pos+3]|ChessDefs.MOVED;
                dstb[pos+3]=0;
              }
              nextmove=moveenum+1;
            } else { // to the left
              if((srcb[pos-1]==0)&&(srcb[pos-2]==0)&&(srcb[pos-3]==0)&&
                 ((srcb[pos-4]&ChessDefs.PIECE)==ChessDefs.ROOK)&&
                 ((srcb[pos-4]&ChessDefs.MOVED)==0)&&
                 ((srcb[pos-4]&ChessDefs.COLOR)==color)&&
                 (!ChessDefs.isAttacked(pos-1, srcb, color^ChessDefs.COLOR))&&
                 (!ChessDefs.isAttacked(pos-2, srcb, color^ChessDefs.COLOR))&&
                 (!ChessDefs.isAttacked(pos-4, srcb, color^ChessDefs.COLOR))) {
                endpos=pos-2;
 
                dstb[pos-1]=srcb[pos-4]|ChessDefs.MOVED;
                dstb[pos-4]=0;
              }
              nextmove=0x100;
            }
          }
        }
        break;

      default:
        throw new InternalError("Piece "+(piece&ChessDefs.PIECE)+" not implemented");
    }

    if(nextmove<0) {
      throw new InternalError("Nextmove uninitialized for move "+Integer.toHexString(move));
    }


    if(pos==63) srcb[NEXTMOVE_POS]=Integer.MAX_VALUE;
    else srcb[NEXTMOVE_POS]=(move&0xFFF00)+nextmove;

    if(endpos<0) return(false);

    dstb[endpos]=dstb[pos]|ChessDefs.MOVED;
    dstb[pos]=ChessDefs.EMPTY;

// clear the MOVED2-flags for all pieces of current color 
    for(i=0; i<64; i++) {
      if(i==endpos) continue;
      if(((dstb[i]&ChessDefs.PIECE)!=0)&&((dstb[i]&ChessDefs.COLOR)==color)) {
        dstb[i]=dstb[i]&(-1^ChessDefs.MOVED2);
      }
    }

    return(true);
  }














  final static int	PAWN_POINTS	=    1000;
  final static int	BISHOP_POINTS	=    3000;
  final static int	KNIGHT_POINTS	=    3000;
  final static int	ROOK_POINTS	=    5000;
  final static int	QUEEN_POINTS	=    9000;
  final static int	KING_POINTS	= 1000000;
  final static int	MATE_POINTS	=  500000;

  final static int	POINT_LIST[]	= { 0, PAWN_POINTS, BISHOP_POINTS,
                                            KNIGHT_POINTS, ROOK_POINTS,
                                            QUEEN_POINTS, KING_POINTS };


  private final static int CHECK_BONUS=10000;	// 1/5 pawn * 50

// score diff counts 1/50
  private final static int KINGFIELDS_PENALTY[] = { -300, -100, -50};

// bonus for promoting pawn to queen
  private final static int PROMOTE_BONUS = 200;


/** Evaluate a given board. The color-bit of the pieces will
 *  get the sign-bit for the result, white pieces are positiv
 *  black one's negative.
 */
  int evaluateBoard(int data[]) {

    int score=0;


    int attacklist[]=new int[64];
    int attackscore=0;
// first step is to calculate an attacklist and an
// attack bonus score. The bonus score is fraction of
// the value difference between the attacking and attacked
// pieces, for KING a fixed check bonus is added 

    for(int pos=0; pos<64; pos++) {
      if(data[pos]==0) continue;

      int piece=data[pos]&ChessDefs.PIECE;
      int color=data[pos]&ChessDefs.COLOR;
      int mask=(color==ChessDefs.WHITE?1:0x100);

      int otherpiece;

      int attackpart=0;

      switch(piece) {
        case ChessDefs.PAWN:
          if(color==ChessDefs.WHITE) {
            int x=(pos&7);
            if(x!=0) { // forward left
              attacklist[pos+7]+=mask;
              otherpiece=data[pos+7]&ChessDefs.PIECE;
              if((otherpiece!=0)&&
                 ((data[pos+7]&ChessDefs.COLOR)!=color)) {
   attackpart=(otherpiece==ChessDefs.KING?CHECK_BONUS:POINT_LIST[otherpiece]);
                score+=PROMOTE_BONUS;
              }
            }
            if(x!=7) { // forward right
              attacklist[pos+9]+=mask;
              otherpiece=data[pos+9]&ChessDefs.PIECE;
              if((otherpiece!=0)&&
                 ((data[pos+9]&ChessDefs.COLOR)!=color)) {
   attackpart=(otherpiece==ChessDefs.KING?CHECK_BONUS:POINT_LIST[otherpiece]);
                score+=PROMOTE_BONUS;
              }
            }
            if(((pos>>3)==6)&&(data[pos+8]==0)) score+=PROMOTE_BONUS;
          } else {

            int x=(pos&7);
            if(x!=0) { // forward left
              attacklist[pos-9]+=mask;
              otherpiece=data[pos-9]&ChessDefs.PIECE;
              if((otherpiece!=0)&&
                 ((data[pos-9]&ChessDefs.COLOR)!=color)) {
   attackpart=(otherpiece==ChessDefs.KING?CHECK_BONUS:POINT_LIST[otherpiece]);
                score-=PROMOTE_BONUS;
              }
            }
            if(x!=7) { // forward right
              attacklist[pos-7]+=mask;
              otherpiece=data[pos-7]&ChessDefs.PIECE;
              if((otherpiece!=0)&&
                 ((data[pos-7]&ChessDefs.COLOR)!=color)) {
   attackpart=(otherpiece==ChessDefs.KING?CHECK_BONUS:POINT_LIST[otherpiece]);
                score-=PROMOTE_BONUS;
              }
            }
            if(((pos>>3)==1)&&(data[pos-8]==0)) score-=PROMOTE_BONUS;
          }
          break;
        case ChessDefs.BISHOP:
          for(int dircnt=1; dircnt<8; dircnt+=2) {
            int x=(pos&7)+dirs[dircnt];
            int y=(pos>>3)+dirs[(dircnt+2)&7];
            while((x>=0)&&(x<8)&&(y>=0)&&(y<8)) {
              int apos=x+(y<<3);
              attacklist[apos]+=mask;
              if((otherpiece=(data[apos]&ChessDefs.PIECE))!=0) {
                if((data[apos]&ChessDefs.COLOR)!=color) {
//                  attackpart=(otherpiece==ChessDefs.KING?CHECK_BONUS:POINT_LIST[otherpiece]-POINT_LIST[piece]);
                  attackpart=(otherpiece==ChessDefs.KING)?CHECK_BONUS:POINT_LIST[otherpiece];
                }
                break;
              }
              x+=dirs[dircnt];
              y+=dirs[(dircnt+2)&7];
            }
          }
          break;
        case ChessDefs.KNIGHT:
          for(int i=0; i<8; i++) {
            int x=(pos&7)+knightmoves[i];
            int y=(pos>>3)+knightmoves[(i+2)&7];

            if((x>=0)&&(x<8)&&(y>=0)&&(y<8)) {
              int apos=x+(y<<3);

              attacklist[apos]+=mask;
              if((otherpiece=(data[apos]&ChessDefs.PIECE))!=0) {
                if((data[apos]&ChessDefs.COLOR)!=color) {
//                  attackpart=(otherpiece==ChessDefs.KING?CHECK_BONUS:POINT_LIST[otherpiece]-POINT_LIST[piece]);
                  attackpart=(otherpiece==ChessDefs.KING)?CHECK_BONUS:POINT_LIST[otherpiece];
                }
              }
            }
          }
          break;
        case ChessDefs.ROOK:
          for(int dircnt=0; dircnt<8; dircnt+=2) {
            int x=(pos&7)+dirs[dircnt];
            int y=(pos>>3)+dirs[(dircnt+2)&7];
            while((x>=0)&&(x<8)&&(y>=0)&&(y<8)) {
              int apos=x+(y<<3);
              attacklist[apos]+=mask;
              if((otherpiece=(data[apos]&ChessDefs.PIECE))!=0) {
                if((data[apos]&ChessDefs.COLOR)!=color) {
//                  attackpart=(otherpiece==ChessDefs.KING?CHECK_BONUS:POINT_LIST[otherpiece]-POINT_LIST[piece]);
                  attackpart=(otherpiece==ChessDefs.KING)?CHECK_BONUS:POINT_LIST[otherpiece];
                }
                break;
              }
              x+=dirs[dircnt];
              y+=dirs[(dircnt+2)&7];
            }
          }
          break;
        case ChessDefs.QUEEN:
          for(int dircnt=0; dircnt<8; dircnt++) {
            int x=(pos&7)+dirs[dircnt];
            int y=(pos>>3)+dirs[(dircnt+2)&7];
            while((x>=0)&&(x<8)&&(y>=0)&&(y<8)) {
              int apos=x+(y<<3);
              attacklist[apos]+=mask;
              if((otherpiece=(data[apos]&ChessDefs.PIECE))!=0) {
                if((data[apos]&ChessDefs.COLOR)!=color) {
//                  attackpart=(otherpiece==ChessDefs.KING?CHECK_BONUS:POINT_LIST[otherpiece]-POINT_LIST[piece]);
                  attackpart=(otherpiece==ChessDefs.KING)?CHECK_BONUS:POINT_LIST[otherpiece];
                }
                break;
              }
              x+=dirs[dircnt];
              y+=dirs[(dircnt+2)&7];
            }
          }
          break;
        case ChessDefs.KING:
          for(int dircnt=1; dircnt<8; dircnt+=2) {
            int x=(pos&7)+dirs[dircnt];
            int y=(pos>>3)+dirs[(dircnt+2)&7];
            if((x>=0)&&(x<8)&&(y>=0)&&(y<8)) {
              int apos=x+(y<<3);
              attacklist[apos]+=mask;
              if((otherpiece=(data[apos]&ChessDefs.PIECE))!=0) {
                if((data[apos]&ChessDefs.COLOR)!=color) {
                  attackpart=POINT_LIST[otherpiece];
                }
              }
            }
          }
          break;
        default:
          throw new InternalError("Field "+data[pos]);
      }


      if(attackpart<0) attackpart=0;	// don't attack if no bonus
      else {
        if(color==ChessDefs.WHITE) attackscore+=attackpart;
        else attackscore-=attackpart;
      }
    }

    score+=attackscore/50;

// check attacklist for kings
    int whiteattacks=0;
    int blackattacks=0;
    for(int pos=0; pos<64; pos++) {
      whiteattacks+=attacklist[pos]&0xFF;
      blackattacks+=(attacklist[pos]>>8)&0xFF;

      if((data[pos]&ChessDefs.PIECE)==ChessDefs.KING) {
// king check, count available fields for this king
        int availfields=0;
        int mask=((data[pos]&ChessDefs.COLOR)==ChessDefs.WHITE)?0xff00:0xff;

        for(int i=0; i<8; i++) {
          int x=(pos&7)+dirs[i];
          int y=(pos>>3)+dirs[(i+2)&0x7];
          if((x>=0)&&(x<8)&&(y>=0)&&(y<8)&&((attacklist[x+(y<<3)]&mask)==0)) {
            availfields++;
          }
        }
        
        if(availfields<KINGFIELDS_PENALTY.length) {
          if((data[pos]&ChessDefs.COLOR)==ChessDefs.WHITE) {
            score+=KINGFIELDS_PENALTY[availfields];
          } else {
            score-=KINGFIELDS_PENALTY[availfields];
          }
        }
      }
    }    


    score+=whiteattacks;
    score-=blackattacks;


    for(int pos=0; pos<64; pos++) {
      int piece;
      if((piece=data[pos]&ChessDefs.PIECE)==ChessDefs.EMPTY) continue;

      int delta=POINT_LIST[piece];
      int color=data[pos]&ChessDefs.COLOR;

/*
      switch(piece) {
        case ChessDefs.PAWN: // no advance bonus
          break;
        case ChessDefs.BISHOP:
          break;
        case ChessDefs.KNIGHT:
          break;
        case ChessDefs.ROOK:
          break;
        case ChessDefs.QUEEN:
          break;
        case ChessDefs.KING:
          break;
        default:
          throw new InternalError("Illegal piece 0x"+Integer.toHexString(data[pos]));
      }
*/
      if(color==ChessDefs.BLACK) score-=delta;
      else score+=delta;
    }


    return(score);
  }








/*
  int evaluateBoard(int data[]) {
    int i, val, sum, j, k;

    sum=0;
    val=0;
    for(i=0; i<64; i++) {
      if((data[i]&ChessDefs.PIECE)==ChessDefs.EMPTY) continue;

      switch(data[i]&ChessDefs.PIECE) {
        case ChessDefs.PAWN:   val=PAWN_POINTS;		break;
        case ChessDefs.BISHOP: val=BISHOP_POINTS;	break;
        case ChessDefs.KNIGHT: val=KNIGHT_POINTS;	break;
        case ChessDefs.ROOK:	val=ROOK_POINTS;	break;
        case ChessDefs.QUEEN:  val=QUEEN_POINTS;	break;
        case ChessDefs.KING:   val=KING_POINTS;		break;
        default:
          throw new InternalError("Illegal piece 0x"+Integer.toHexString(data[i]));
      }

      if((data[i]&ChessDefs.COLOR)==ChessDefs.BLACK) val=-val;

      sum+=val;
    }

    return(sum);
  }
*/





/** Create a move object from a move val
 */
  private Move createMove(int val, int piece) {
    int from=val>>8;
    int to=-1;
    int newpiece=0;

    int variant=val&0xFF;
    int color=piece&ChessDefs.COLOR;

    int fx=from&7;
    int fy=from>>3;

    switch(piece&ChessDefs.PIECE) {
      case ChessDefs.PAWN:
        boolean toknight=false;

        int direction=((color==ChessDefs.WHITE)?8:-8);
        switch(variant) {
          case 4: // 1 forwd+K
            toknight=true;
          case 0: // 1 forwd
            to=from+direction;
            break;
          case 5:
            toknight=true;
          case 1: // attack right
            to=from+direction+1;
            break;
          case 6:
            toknight=true;
          case 2: // attack left
            to=from+direction-1;
            break;
          case 3: // 2 forward
            to=from+direction+direction;
            break;
        }

        if(((to>>3)==0)||((to>>3)==7)) { // promote pawn
          newpiece=(toknight ? ChessDefs.KNIGHT : ChessDefs.QUEEN);
        }
        break;


      case ChessDefs.BISHOP:
        direction=(variant>>2)|1;
        int dist=variant&0x7;
        to=from+(1+dist)*dirs[direction]+
                (((1+dist)*dirs[(direction+2)&0x7])<<3);
        break;
      case ChessDefs.KNIGHT:
        to=from+knightmoves[variant]+(knightmoves[(variant+2)&7]<<3);
        break;

      case ChessDefs.ROOK:
        direction=(variant>>2)&(-2);
        dist=variant&0x7;
        to=from+(1+dist)*dirs[direction]+
                (((1+dist)*dirs[(direction+2)&0x7])<<3);
        break;
      case ChessDefs.QUEEN:
        direction=variant>>3;
        dist=variant&0x7;
        to=from+(1+dist)*dirs[direction]+
                (((1+dist)*dirs[(direction+2)&0x7])<<3);
        break;
      case ChessDefs.KING:
        if(variant<8) {
          to=from+dirs[variant]+(dirs[(variant+2)&0x7]<<3);
        } else if(variant==8) to=from+2;
        else if(variant==9) to=from-2;
        break;
      default:
        throw new InternalError("Illegal piece "+piece);
    }

    return(new Move(from, to, newpiece));
  }



}




