package fips.game.jchess;

import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Point;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.logging.Logger;
import java.util.logging.Level;

import javax.swing.JComponent;



/** Simple GUI displaying a chess board. The board notifies
 *  listeners if a piece was dragged.
 */
public class BoardGUI extends JComponent
                      implements java.awt.event.MouseMotionListener,
                                 java.awt.event.MouseListener,
                                 BoardListener {


  public final static String piecenames[]={
      "fieldw.gif", "pawnw.gif", "bishopw.gif", "knightw.gif",
      "rookw.gif", "queenw.gif", "kingw.gif", 
      "fieldb.gif", "pawnb.gif", "bishopb.gif", "knightb.gif",
      "rookb.gif", "queenb.gif", "kingb.gif" 
  };


  Board			board;
  int			olddata[];

/** The color who has normal view
 */
  int			view;

  Image			pieces[];

  int			cellsize;


/** Image for buffering the screen cause the graphics context
 *  of the canvas isn't readable, it isn't possible to grab
 *  a image from the screen.
 */
  Image			buffer;

/** If true, board is editable.
 */
  boolean		editable;

/** If false, board currently in edit mode, in play mode
 *  otherwise. In play mode only piece drags of correct
 *  color are allowed, in edit mode any clicks or drags.
 */
  boolean		playmode;

  ArrayList		listeners;

  Point			click_point;
  Point			click_offset;

/** If not null, board in drag mode (drawed with dragging
 *  image missing and drag_image points to piece being moved.
 */
  Image			drag_image;
  Point			drag_point;

/** If true, piece drag was triggered from outside. This is
 *  usefull to allows users to select pieces from outside
 *  the board and put them onto the board in setup game mode.
 */
  boolean		extern_drag;


  Logger		logger;


  public BoardGUI(String series, Board board) {
    super();

    logger=fips.util.LogManager.getLogger("jchess.BoardGUI");

    this.board=board;
    board.addBoardListener(this);

    view=ChessDefs.WHITE;

//
// load the images
//

    pieces=new Image[ChessDefs.PIECE<<1];
    java.io.InputStream in;

    for(int i=0; i<pieces.length; i++) {
      try {
        in=getClass().getResourceAsStream(series+"/"+piecenames[i]);

        byte buf[]=new byte[in.available()];
        for(int m=0; m<buf.length; ) {
          int n=in.read(buf, m, buf.length-m);
          if(n>0) m+=n;
        }

        //ARM ("#: loaded "+buf.length+" bytes from "+piecenames[i]);

        pieces[i]=java.awt.Toolkit.getDefaultToolkit().createImage(buf);
        prepareImage(pieces[i], this);
        while(pieces[i].getWidth(this)<=0) {
          try {
            Thread.sleep(100);
          } catch (InterruptedException ie) {}
        }
      } catch (Exception e) {
        //ERROR ("#: can't load image for piece "+piecenames[i]);
        //ERROR e.printStackTrace();
      }
    }

    cellsize=pieces[0].getWidth(null);
    for(int i=0; i<pieces.length; i++) {
      if((cellsize!=pieces[i].getWidth(null)) ||
         (cellsize!=pieces[i].getHeight(null))) {
        //ERROR ("#: picture "+piecenames[i]+" has invalid size.");
        throw new IllegalArgumentException("Picture "+piecenames[i]+
                                           " has invalid size.");
      }
    }

    setSize(cellsize<<3, cellsize<<3);
    setPreferredSize(new Dimension(cellsize<<3, cellsize<<3));

    editable=false;
    playmode=false;
    listeners=new ArrayList();

    addMouseMotionListener(this);
    addMouseListener(this);
  }



  public int getView() { return(view); }

  public void setView(int color) {
    logger.fine(".setView(color="+color+"), old="+view);
    if((color!=ChessDefs.WHITE)&&(color!=ChessDefs.BLACK)) {
      throw new IllegalArgumentException();
    }

    if(color!=view) {
      view=color;
      olddata=null;
      buffer=null;
      repaint();
    }
  }


  public void setEditable(boolean val) {
    editable=val;
  }

  public void setPlayMode(boolean val) {
    playmode=val;
  }


  public void boardChanged(Board board) {
    logger.fine("boardChanged()");

    this.board=board;
    paintBuffer(board.getData());
    repaint();
  }



//**************************************************************************//
//
//			Graphic update
//
//****************************************


/** Draw a certain piece at pos
 */
  private void drawPiece(Graphics g, int piece, int pos) {
    int x=pos&7;
    int y=pos>>3;
    if(view==ChessDefs.WHITE) y=7-y;
    else x=7-x;

    g.drawImage(pieces[((x^y)&1)==0?0:ChessDefs.PIECE],
                x*cellsize, y*cellsize, null);
    if((piece&ChessDefs.PIECE)==0) return;

    piece&=(ChessDefs.PIECE|ChessDefs.COLOR);
    if((piece&ChessDefs.COLOR)!=0) {
      piece=(piece&ChessDefs.PIECE)+ChessDefs.PIECE;
    }
    logger.finer("drawing "+piece+" at "+x+"x"+y);
    g.drawImage(pieces[piece], x*cellsize, y*cellsize, null);
  }



/** Paint the board data into the buffer.
 */
  private void paintBuffer(int data[]) {
    logger.fine("paintBuffer()");

    if(buffer==null) {
      buffer=createImage(cellsize<<3, cellsize<<3);
      olddata=null;
    }


    Graphics g=null;

    if((olddata==null)||(!java.util.Arrays.equals(olddata, data))) {
      for(int x=0; x<8; x++) {
        for(int y=0; y<8; y++) {
        
          if((data==null)||(olddata==null)||
             (olddata[x+(y<<3)]!=data[x+(y<<3)])) {
// draw the field tile

            if(g==null) g=buffer.getGraphics();
            drawPiece(g, (data==null)?0:data[x+(y<<3)], x+(y<<3));
          }
        }
      }
    }
    olddata=data;
  }



  public void paintComponent(Graphics g) {
    logger.fine(".paintComponent()");

    if(cellsize<=0) {
// not yet loaded
      return;
    }

    if(buffer==null) paintBuffer(board.getData());

    g.drawImage(buffer, 0, 0, null);
    if((drag_image!=null)&&(drag_point!=null)) {
      g.drawImage(drag_image, drag_point.x-click_offset.x,
                  drag_point.y-click_offset.y, null);
    }
  }






//**************************************************************************//
//									    //
//				MouseListener				    //
//									    //
//**************************************************************************//


  public void mouseClicked(MouseEvent e) {
    logger.severe("Not yet ...");

  }
  public void mouseEntered(MouseEvent e) {}
  public void mouseExited(MouseEvent e) {}


/** Select a piece and init the buffers.
 */
  public void mousePressed(MouseEvent e) {
    logger.fine("mousePressed() at "+e.getX()+"x"+e.getY());

    drag_image=null;

    if(!editable) return;
    click_point=e.getPoint();
    int pos=pointToPos(click_point);

    int data[]=board.getData();
    if((pos<0)||(data==null)||(data[pos]==0)||
       ((playmode)&&((data[pos]&ChessDefs.COLOR)!=board.getMover()))) {
      logger.finer("Not valid click position pos="+pos);
      click_point=null;
      return;
    }

    click_offset=new Point(click_point.x%cellsize,
                           click_point.y%cellsize);

/*
    buffer.getGraphics().drawImage(pieces[(((x+y)&0x1)^0x1)*ChessDefs.PIECE],
                                   x*cellsize, (7-y)*cellsize, null);

*/
  }



/** If drag from inside and click_point set then valid drag
 */
  public void mouseReleased(MouseEvent e) {
    logger.fine("mouseReleased() ");

    int startpos=click_point==null?-1:pointToPos(click_point);
    int endpos=pointToPos(e.getPoint());

    if(!editable) {
      logger.finer("No action if not in editmode");

    } else if(extern_drag) {
// drag from outside  
      logger.severe("Drag from outside not yet impl.");

    } else if(click_point==null) {
// illegal inner drag -> ignore
    } else if(click_point.equals(e.getPoint())) {
      logger.finer("Click detected, ignoring it");
    } else {

// drag from inside
      if(startpos==endpos) {
// only in normal edit mode allowed
        if(!playmode) {
// restore buffer (by writing real piece at startpos)

// send click

          logger.severe("Not yet: like clicked");

        }
      } else {

// real drag
        boolean nopaint=true;
        int data[]=board.getData();

        if(playmode) {
// check validity of target pos
          if((data[endpos]==0)||
             ((data[endpos]&ChessDefs.COLOR)!=(data[startpos]&ChessDefs.COLOR))) {
            nopaint=firePieceDragged(startpos, endpos);
          } else nopaint=false;
        } else {
// just drag it
          nopaint=firePieceDragged(startpos, endpos);
        }

        if(nopaint) {
// put piece onto target field and send repaint
          logger.fine("Drag valid, updating buffer");
          paintBuffer(board.getData());
        } else {
// revert buffer
          logger.fine("Drag invalid, reverting buffer");
          if(olddata!=null) { olddata[startpos]=-1; olddata[endpos]=-1; }
          paintBuffer(data);
        }
        repaint();
      }
    }

    click_point=null;
    drag_image=null;
    extern_drag=false;
  }





// ************************************************************************* //
//									     //
//			MouseMotionListener impl			     //
//									     //
// ************************************************************************* //




  public void mouseMoved(MouseEvent e) {}
  
  public void mouseDragged(MouseEvent e) {
    logger.fine("mouseDragged()");

// check for illegal inner drag
    if((!extern_drag)&&(click_point==null)) return;

// write emtpy field at startpos into buffer,
    if(drag_image==null) {
      int pos=pointToPos(click_point);
      int piece=board.getData()[pos];
      drag_image=pieces[(piece&ChessDefs.PIECE)+
                        ((piece&ChessDefs.COLOR)!=0?ChessDefs.PIECE:0)];
      drawPiece(buffer.getGraphics(), 0, pos);
    }

    drag_point=e.getPoint();
    repaint();
  }



  private int pointToPos(Point p) {
    int x=p.x/cellsize;
    int y=p.y/cellsize;
    if(view==ChessDefs.WHITE) y=7-y;
    else  x=7-x;

    if((x<0)||(x>7)||(y<0)||(y>7)) return(-1);
    return(x+(y<<3));
  }


//**************************************************************************//
//
//			BoardMouseListener support
//
//***************************




  public void addBoardMouseListener(BoardMouseListener bml) {
    if(bml!=null) listeners.add(bml);
  }

  public void removeBoardMouseListener(BoardMouseListener bml) {
    listeners.remove(bml);
  }


  public boolean firePieceDragged(int from, int to) {
    boolean nopaint=true;

    for(int i=0; i<listeners.size(); i++) {
      nopaint&=((BoardMouseListener)listeners.get(i)).pieceDragged(from, to);
    }
    return(nopaint);
  }


}



