package fips.game.jchess;

import java.util.ArrayList;
import java.util.logging.Level;
import java.util.logging.Logger;

/** Class containing the board.
 *
 *  Normal use:
 *   1.)  reset / setup / sitWhite / sitBlack
 *   2.)  start game
 *   3.)  end / stop
 *
 */
public class Board {

/** List of all positions in this game. The first element is
 *  the original board.
 */
  ArrayList		history;

/** Current board data, also in history.
 */
  int			data[];

  int			moving;

  ArrayList		boardlisteners;

  Player		white;
  Player		black;

/** If true game allready started.
 */
  boolean		started;

  Logger		logger;

  public Board() {
    logger=fips.util.LogManager.getLogger("jchess.Board");
    history = new ArrayList();
    boardlisteners=new ArrayList();
    data=new int[64];
    reset();
  }

  public int[] getData() { return((int[])data.clone()); }

  public int getMover() {
    logger.fine(".getMover(): "+moving);
    return(moving);
  }




/** Set the board back to the beginning of the game. The
 *  history is cleared. The state of the game is then
 */
  public void reset() {
    if(started) throw new IllegalStateException("Stop game first");
    moving=ChessDefs.WHITE;
    history.clear();

    data=new int[64];

    for(int i=0; i<8; i++) {
      data[8+i]=ChessDefs.PAWN|ChessDefs.WHITE;
      data[48+i]=ChessDefs.PAWN|ChessDefs.BLACK;
    }
    for(int i=0; i<8; i++) {
      data[i]=ChessDefs.WHITE; data[56+i]=ChessDefs.BLACK;
    }

    data[0]|=ChessDefs.ROOK; data[7]|=ChessDefs.ROOK;
    data[1]|=ChessDefs.KNIGHT; data[6]|=ChessDefs.KNIGHT;
    data[2]|=ChessDefs.BISHOP; data[5]|=ChessDefs.BISHOP;
    data[3]|=ChessDefs.QUEEN; data[4]|=ChessDefs.KING;

    data[56]|=ChessDefs.ROOK; data[63]|=ChessDefs.ROOK;
    data[57]|=ChessDefs.KNIGHT; data[62]|=ChessDefs.KNIGHT;
    data[58]|=ChessDefs.BISHOP; data[61]|=ChessDefs.BISHOP;
    data[59]|=ChessDefs.QUEEN; data[60]|=ChessDefs.KING;


/*
    data[56]=ChessDefs.ROOK|ChessDefs.BLACK;
    data[4]=ChessDefs.KING|ChessDefs.WHITE;
    data[60]=ChessDefs.KING|ChessDefs.BLACK;
*/

    notifyBoardListeners();
  }



  public void setPiece(int position, int piece, int color) {
    if(started) throw new IllegalArgumentException();
    throw new InternalError();
  }


  public boolean gameStarted() { return(started); }


/** Start the game / timer
 */
  public void start() {
    if(started) throw new IllegalArgumentException();

    if(history.size()==0) history.add(data);

    logger.severe("No timer support yet");
    started=true;
  }

  public void stop() {
  }
  

/** Set the player for one color.
 */
  public void setPlayer(Player player, int color)
                        throws IllegalStateException {
    if(color==ChessDefs.WHITE) {
      if(player==white) return;
      if(white!=null) {
// unregister old player
        player.sit(null, color);
        removeBoardListener(player);
      }
      white=player;
    } else if(color==ChessDefs.BLACK) {
      if(player==black) return;
      if(black!=null) {
// unregister old player
        player.sit(null, color);
        removeBoardListener(player);
      }
      black=player;
      player.sit(this, color);
      player.boardChanged(this);
    } else {
      throw new IllegalArgumentException();
    }

    player.sit(this, color);
    player.boardChanged(this);
    addBoardListener(player);
  }


  public void makeMove(Player player, Move move)
                       throws IllegalArgumentException {
    if(!started) throw new IllegalStateException();

// check the move
    int newdata[]=null;
    try {
      newdata=checkMove(player, move);
    } catch (IllegalArgumentException e) {
      logger.fine("Move "+move.from+" to "+move.to+" not accepted: "+e);
      logger.log(Level.FINER, "Cause ", e);
      throw(e);
    }

// make it
    history.add(data);
    data=newdata;
    moving^=ChessDefs.COLOR;

    notifyBoardListeners();
  }





  private int[] checkMove(Player p, Move move) throws IllegalArgumentException {
    if(p!=(moving==ChessDefs.WHITE?white:black)) throw new IllegalArgumentException("Wrong player");

    int piece=data[move.from];
    int color=piece&ChessDefs.COLOR;
    if(((piece&ChessDefs.PIECE)==0)||(color!=moving)) throw new IllegalArgumentException("No piece at "+move.from);

    if(((data[move.to]&ChessDefs.PIECE)!=0)&&
       ((data[move.to]&ChessDefs.COLOR)==moving)) throw new IllegalArgumentException("Can't attack own piece");


// from, to, player ok -> perform move
    int newdata[]=new int[64];
    System.arraycopy(data, 0, newdata, 0, 64);

    for(int i=0; i<64; i++) { // clear moved2 flag from own color
      if((newdata[i]==0)||((newdata[i]&ChessDefs.COLOR)!=moving)) continue;
      newdata[i]&=(-1^ChessDefs.MOVED2);
    }


    int fx=move.from&7; int fy=move.from>>3;
    int tx=move.to&7; int ty=move.to>>3;
    int dx=tx-fx; int dy=ty-fy;

    switch(piece&ChessDefs.PIECE) {
      case ChessDefs.PAWN:
        if((dx!=1)&&(dx!=0)&&(dx!=-1)) throw new IllegalArgumentException("Pawn may only move 1 forward, left and right");
        if((dx==0)&&((data[move.to]&ChessDefs.PIECE)!=0)) {
          throw new IllegalArgumentException("Pawn can't attack directly");
        }

        if(((color==ChessDefs.WHITE)&&(dy!=1)&&(dy!=2))||
           ((color==ChessDefs.BLACK)&&(dy!=-1)&&(dy!=-2))) {
          throw new IllegalArgumentException();
        }

        if(dx!=0) {
          if((dy!=1)&&(dy!=-1)) throw new IllegalArgumentException();

          if((data[move.to]&ChessDefs.PIECE)!=0) {
            if((data[move.to]&ChessDefs.COLOR)==color) {
              throw new IllegalArgumentException();
            }
          } else {
// en passant??
            if(((data[move.from+dx]&ChessDefs.MOVED2)!=0)&&
               ((data[move.from+dx]&ChessDefs.COLOR)!=color)) {
// valid en passant, remove other pawn
              newdata[move.from+dx]=0;
              logger.finest("En passant accepted");
            } else {
              throw new IllegalArgumentException();
            }
          }
        }

        if(color==ChessDefs.WHITE) {
          if(dy==2) {
            if((fy!=1)||(dx!=0)||((data[move.from+8]&ChessDefs.PIECE)!=0)) {
              throw new IllegalArgumentException();
            }
            newdata[move.from]|=ChessDefs.MOVED2;
          } else { // 1 forward already checked

          }
        } else {
          if(dy==-2) {
            if((fy!=6)||(dx!=0)||((data[move.from-8]&ChessDefs.PIECE)!=0)) {
              throw new IllegalArgumentException();
            }
            newdata[move.from]|=ChessDefs.MOVED2;
          } else { // 1 forward already checked
          }
        }
        if((ty==0)||(ty==7)) {
          newdata[move.from]&=(~ChessDefs.PIECE);
          newdata[move.from]|=(move.newpiece&ChessDefs.PIECE);
          if((newdata[move.from]&ChessDefs.PIECE)==0) {
            logger.finest("Default promote to queen");
            newdata[move.from]|=ChessDefs.QUEEN;
          }
        }
        break;

      case ChessDefs.BISHOP:
        if((dx!=dy)&&(dx!=-dy)) throw new IllegalArgumentException();
// make sure that way to dx,dy is free
        int dir=(dx<0?-1:1)+(dy<0?-8:8);
        for(int i=move.from+dir; i!=move.to; i+=dir) {
          if(data[i]!=0) throw new IllegalArgumentException();
        }

        break;

      case ChessDefs.KNIGHT:
        if(dx<0) dx=-dx;
        if(dy<0) dy=-dy;
        if(((dx!=1)&&(dy!=1))||((dx!=2)&&(dy!=2))||
           ((dx!=dy+1)&&(dy!=dx+1))) throw new IllegalArgumentException();
        break;

      case ChessDefs.ROOK:
        if((dx!=0)&&(dy!=0)) throw new IllegalArgumentException();
        if(dx!=0) dx=dx<0?-1:1;
        if(dy!=0) dy=dy<0?-8:8;
        for(int i=move.from+dx+dy; i!=move.to; i+=dx+dy) {
          if(data[i]!=0) throw new IllegalArgumentException();
        }

        break;
      case ChessDefs.QUEEN:
       if((dx!=0)&&(dy!=0)&&(dy!=dx)&&(dy!=-dx)) throw new IllegalArgumentException();
        if(dx!=0) dx=dx<0?-1:1;
        if(dy!=0) dy=dy<0?-8:8;
        for(int i=move.from+dx+dy; i!=move.to; i+=dx+dy) {
          if(data[i]!=0) throw new IllegalArgumentException();
        }
        break;

      case ChessDefs.KING:
        if((dx==2)||(dx==-2)) {
// rochade??
          int rookp=(dx<0?0:7)+(fy<<3);
          if(((data[move.to]&ChessDefs.MOVED)!=0)||(fx!=4)||(dy!=0)||
             ((fy!=0)&&(fy!=7))||
             ((data[rookp]&ChessDefs.PIECE)!=ChessDefs.ROOK)||
             ((data[rookp]&ChessDefs.COLOR)!=color)) {
            throw new IllegalArgumentException();
          }
          if(dx<0) {
// check that fields between king and rook empty and not attacked
            for(int i=0; i<5; i++) {
              if((i!=0)&&(i!=4)&&(data[move.from-i]!=0)) throw new IllegalArgumentException();
              if(ChessDefs.isAttacked(move.from-i, data, color^ChessDefs.COLOR)) throw new IllegalArgumentException();
            }
            newdata[move.from-1]=newdata[rookp]|ChessDefs.MOVED;
            newdata[rookp]=0;
          } else {
// check that fields between king and rook empty and not attacked
            for(int i=0; i<4; i++) {
              if((i!=0)&&(i!=3)&&(data[move.from+i]!=0)) throw new IllegalArgumentException();
              if(ChessDefs.isAttacked(move.from+i, data, color^ChessDefs.COLOR)) throw new IllegalArgumentException();
            }
            newdata[move.from+1]=newdata[rookp]|ChessDefs.MOVED;
            newdata[rookp]=0;
          }

        } else if((dx<-1)||(dx>1)||(dy<-1)||(dy>1)) {
          throw new IllegalArgumentException();
        }
        break;
      default:
        throw new IllegalArgumentException("Illegal piece "+piece);
    }

    newdata[move.to]=newdata[move.from]|ChessDefs.MOVED;
    newdata[move.from]=0;

    if((color==ChessDefs.WHITE)&&(ty==5)&&
       ((newdata[move.to-8]&ChessDefs.MOVED2)!=0)) newdata[move.to-8]=0;
    if((color==ChessDefs.BLACK)&&(ty==2)&&
       ((newdata[move.to+8]&ChessDefs.MOVED2)!=0)) newdata[move.to+8]=0;

    logger.fine("Made move, doing mate check");

    for(int i=0; i<64; i++) {
      if(((newdata[i]&ChessDefs.COLOR)==moving)&&
         ((newdata[i]&ChessDefs.PIECE)==ChessDefs.KING)) {
        if(ChessDefs.isAttacked(i, newdata, moving^ChessDefs.COLOR)) {
          throw new IllegalArgumentException("Protect your king");
        }
        break;
      }
    }

    return(newdata);
  }






// ************************************************************************* //
//			Listener methods				     //
// ******


  public void addBoardListener(BoardListener bl) {
    if(bl!=null) boardlisteners.add(bl);
  }

  public void removeBoardListener(BoardListener bl) {
    boardlisteners.remove(bl);
  }

  private void notifyBoardListeners() {
    int i;
    for(i=0; i<boardlisteners.size(); i++) {
      ((BoardListener)boardlisteners.get(i)).boardChanged(this);
    }
  }


}
